// Content script that runs automatically on AutoTrader pages
// Extracts car details and sends them to the API

const API_URL = 'https://www.carpricetracker.com/api/save-car/';

// Function to extract car details from the page
function extractCarDetails() {
  const debug = [];

  // Extract price
  const priceElement = document.querySelector('[data-testid="advert-price"]') ||
                      document.querySelector('[data-testid="price"]') ||
                      document.querySelector('.product-card-pricing__price');
  const price = priceElement ? priceElement.textContent.trim() : null;
  debug.push(`Price element found: ${!!priceElement}, value: ${price}`);

  // Extract car title/name
  const titleElement = document.querySelector('h1') ||
                      document.querySelector('[data-testid="advert-heading"]');
  const title = titleElement ? titleElement.textContent.trim() : null;
  debug.push(`Title element found: ${!!titleElement}, value: ${title}`);

  // Extract make and model
  let make = null;
  let model = null;

  const makeElement = document.querySelector('[data-testid="make"]');
  const modelElement = document.querySelector('[data-testid="model"]');

  if (makeElement) {
    make = makeElement.textContent.trim();
    debug.push(`Make from element: ${make}`);
  }
  if (modelElement) {
    model = modelElement.textContent.trim();
    debug.push(`Model from element: ${model}`);
  }

  // Fallback: extract from title
  if (!make && !model && title) {
    const titleParts = title.split(' ');
    if (titleParts[0] && /^\d{4}$/.test(titleParts[0])) {
      titleParts.shift();
    }
    if (titleParts.length >= 2) {
      make = titleParts[0];
      model = titleParts.slice(1).join(' ');
      debug.push(`Make/Model from title: ${make} ${model}`);
    }
  }

  // Extract primary image
  let imageUrl = null;
  const heroImg = document.querySelector('[data-testid="hero-image"] img') ||
                  document.querySelector('[data-testid="gallery-image-0"] img') ||
                  document.querySelector('img[alt*="main"]') ||
                  document.querySelector('picture img') ||
                  document.querySelector('[class*="gallery"] img:first-of-type');

  if (heroImg) {
    imageUrl = heroImg.src || heroImg.getAttribute('data-src');
    debug.push(`Image found: ${imageUrl?.substring(0, 50)}...`);
  } else {
    debug.push('No image found');
  }

  // Extract mileage
  let mileage = null;
  let mileageRaw = null;
  let mileageElement = null;
  const allParagraphs = Array.from(document.querySelectorAll('p'));
  const mileageLabel = allParagraphs.find(p => p.textContent.trim() === 'Mileage');

  if (mileageLabel) {
    let nextElement = mileageLabel.nextElementSibling;
    if (!nextElement && mileageLabel.parentElement) {
      nextElement = mileageLabel.parentElement.nextElementSibling;
    }
    if (!nextElement) {
      const container = mileageLabel.parentElement;
      if (container) {
        const containerPs = Array.from(container.querySelectorAll('p'));
        const labelIndex = containerPs.indexOf(mileageLabel);
        if (labelIndex >= 0 && labelIndex < containerPs.length - 1) {
          nextElement = containerPs[labelIndex + 1];
        }
      }
    }
    if (nextElement) {
      mileageElement = nextElement;
      debug.push(`Mileage found via "Mileage" label`);
    }
  }

  if (!mileageElement) {
    mileageElement = document.querySelector('[data-testid="mileage"]') ||
                    document.querySelector('[class*="mileage"]') ||
                    Array.from(document.querySelectorAll('li, span, div, button')).find(el => {
                      const text = el.textContent.toLowerCase();
                      return /\d+[,\s]?\d*\s*miles/.test(text) && text.length < 50;
                    });
    if (mileageElement) {
      debug.push(`Mileage found via fallback selector`);
    }
  }

  if (mileageElement) {
    mileageRaw = mileageElement.textContent.trim();
    const mileageMatch = mileageRaw.match(/(\d{1,3})[,\s]?(\d{3})/);
    if (mileageMatch) {
      mileage = parseInt(mileageMatch[1] + mileageMatch[2], 10);
      debug.push(`Mileage parsed: "${mileageRaw}" → ${mileage}`);
    } else {
      const singleMatch = mileageRaw.match(/(\d+)/);
      if (singleMatch) {
        mileage = parseInt(singleMatch[1], 10);
        debug.push(`Mileage (single number): "${mileageRaw}" → ${mileage}`);
      }
    }
  } else {
    debug.push('Mileage: No element found');
  }

  // Extract year
  let year = null;
  const registrationLabel = allParagraphs.find(p => p.textContent.trim() === 'Registration');

  if (registrationLabel) {
    let nextElement = registrationLabel.nextElementSibling;
    if (!nextElement && registrationLabel.parentElement) {
      nextElement = registrationLabel.parentElement.nextElementSibling;
    }
    if (!nextElement) {
      const container = registrationLabel.parentElement;
      if (container) {
        const containerPs = Array.from(container.querySelectorAll('p'));
        const labelIndex = containerPs.indexOf(registrationLabel);
        if (labelIndex >= 0 && labelIndex < containerPs.length - 1) {
          nextElement = containerPs[labelIndex + 1];
        }
      }
    }
    if (nextElement) {
      const regText = nextElement.textContent.trim();
      const yearMatch = regText.match(/\b(19|20)\d{2}\b/);
      if (yearMatch) {
        year = parseInt(yearMatch[0], 10);
        debug.push(`Year found via "Registration" label: "${regText}" → ${year}`);
      }
    }
  }

  if (!year) {
    const yearElement = document.querySelector('[data-testid="year"]') ||
                       document.querySelector('[class*="year"]') ||
                       Array.from(document.querySelectorAll('li, span')).find(el => {
                         const text = el.textContent;
                         return /^(19|20)\d{2}$/.test(text.trim()) && text.length < 10;
                       });

    if (yearElement) {
      const yearText = yearElement.textContent.trim();
      const yearMatch = yearText.match(/\b(19|20)\d{2}\b/);
      if (yearMatch) {
        year = parseInt(yearMatch[0], 10);
        debug.push(`Year element found via fallback: "${yearText}" → ${year}`);
      }
    } else {
      debug.push('Year: No element found');
    }
  }

  // Extract registration
  const regElement = document.querySelector('[data-testid="registration"]');
  const registration = regElement ? regElement.textContent.trim() : null;
  debug.push(`Registration: ${registration}`);

  // Extract engine size
  let engineSize = null;
  const engineLabel = allParagraphs.find(p => p.textContent.trim() === 'Engine');
  debug.push(`Engine label search: Found ${engineLabel ? 'YES' : 'NO'}`);

  if (engineLabel) {
    debug.push(`Engine label HTML: ${engineLabel.outerHTML.substring(0, 100)}`);
    debug.push(`Engine label parent HTML: ${engineLabel.parentElement?.outerHTML.substring(0, 150)}`);

    let nextElement = engineLabel.nextElementSibling;
    debug.push(`Direct nextElementSibling: ${nextElement ? 'YES' : 'NO'}`);

    if (!nextElement && engineLabel.parentElement) {
      nextElement = engineLabel.parentElement.nextElementSibling;
      debug.push(`Parent's nextElementSibling: ${nextElement ? 'YES' : 'NO'}`);
    }

    if (!nextElement) {
      const container = engineLabel.parentElement;
      if (container) {
        const containerPs = Array.from(container.querySelectorAll('p'));
        const labelIndex = containerPs.indexOf(engineLabel);
        debug.push(`Container has ${containerPs.length} paragraphs, label at index ${labelIndex}`);
        if (labelIndex >= 0 && labelIndex < containerPs.length - 1) {
          nextElement = containerPs[labelIndex + 1];
          debug.push(`Found via container index: YES`);
        }
      }
    }

    if (nextElement) {
      engineSize = nextElement.textContent.trim();
      debug.push(`Engine size extracted: "${engineSize}"`);
      debug.push(`Engine value element HTML: ${nextElement.outerHTML.substring(0, 100)}`);
    } else {
      debug.push(`Engine size: Could not find value element`);
    }
  } else {
    debug.push('Engine size: No "Engine" label found');
    // Try alternative approach - look for the specific class structure
    const engineContainers = Array.from(document.querySelectorAll('div')).filter(div => {
      const text = div.textContent.trim();
      return text.startsWith('Engine') && text.length < 50;
    });
    debug.push(`Alternative search found ${engineContainers.length} potential engine containers`);

    if (engineContainers.length > 0) {
      const container = engineContainers[0];
      debug.push(`First container text: "${container.textContent.trim()}"`);
      const ps = Array.from(container.querySelectorAll('p'));
      debug.push(`Container has ${ps.length} paragraphs`);
      ps.forEach((p, idx) => {
        debug.push(`  P[${idx}]: "${p.textContent.trim()}"`);
      });

      // Try to find the value in the second paragraph
      if (ps.length >= 2) {
        engineSize = ps[1].textContent.trim();
        debug.push(`Engine size from alternative method: "${engineSize}"`);
      }
    }
  }

  return {
    price,
    title,
    make,
    model,
    imageUrl,
    mileage,
    mileageRaw,
    year,
    registration,
    engineSize,
    url: window.location.href,
    debug: debug.join('\n')
  };
}

// Function to send data to API
async function sendToAPI(carDetails) {
  // Send to API even if some fields are missing - let the backend handle validation
  try {
    console.log('[AutoTrader Tracker] Sending data to API:', carDetails);
    console.log('[AutoTrader Tracker] Debug info:\n' + carDetails.debug);

    const response = await fetch(API_URL, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(carDetails)
    });

    const result = await response.json();

    if (result.success) {
      console.log('[AutoTrader Tracker] Data saved successfully:', result);

      // Store car ID for popup to use
      if (result.car_id) {
        chrome.storage.local.set({ lastCarId: result.car_id });
      }

      // Show a subtle notification badge on the extension icon
      if (chrome.action && chrome.action.setBadgeText) {
        chrome.action.setBadgeText({ text: '✓' });
        chrome.action.setBadgeBackgroundColor({ color: '#27ae60' });
        setTimeout(() => {
          chrome.action.setBadgeText({ text: '' });
        }, 3000);
      }
    } else {
      console.error('[AutoTrader Tracker] Failed to save:', result.error);
    }
  } catch (error) {
    console.error('[AutoTrader Tracker] Error sending to API:', error);
  }
}

// Wait for page to load, then extract and send data
function init() {
  // Run on ALL AutoTrader pages
  console.log('[AutoTrader Tracker] On AutoTrader page, extracting data...');

  // Wait a bit for dynamic content to load
  setTimeout(() => {
    const carDetails = extractCarDetails();

    // Only send if we found at least a price (indicates it's likely a car listing)
    if (carDetails.price) {
      sendToAPI(carDetails);
    } else {
      console.log('[AutoTrader Tracker] No price found, likely not a car listing page');
    }
  }, 2000);
}

// Run when page loads
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}
