// Function to extract car details from AutoTrader page
async function extractCarDetails() {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

  // Check if we're on an AutoTrader page
  if (!tab.url || !tab.url.includes('autotrader.co.uk')) {
    return {
      error: 'Please navigate to an AutoTrader UK car listing page'
    };
  }

  // Inject script to extract data from the page
  const results = await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    function: () => {
      const debug = [];

      // Extract price - use the correct data-testid
      const priceElement = document.querySelector('[data-testid="advert-price"]') ||
                          document.querySelector('[data-testid="price"]') ||
                          document.querySelector('.product-card-pricing__price');
      const price = priceElement ? priceElement.textContent.trim() : null;
      debug.push(`Price element found: ${!!priceElement}, value: ${price}`);

      // Extract car title/name
      const titleElement = document.querySelector('h1') ||
                          document.querySelector('[data-testid="advert-heading"]');
      const title = titleElement ? titleElement.textContent.trim() : null;
      debug.push(`Title element found: ${!!titleElement}, value: ${title}`);

      // Extract make and model from title or separate elements
      let make = null;
      let model = null;

      const makeElement = document.querySelector('[data-testid="make"]');
      const modelElement = document.querySelector('[data-testid="model"]');

      if (makeElement) {
        make = makeElement.textContent.trim();
        debug.push(`Make from element: ${make}`);
      }
      if (modelElement) {
        model = modelElement.textContent.trim();
        debug.push(`Model from element: ${model}`);
      }

      // Fallback: extract from title
      if (!make && !model && title) {
        const titleParts = title.split(' ');
        // Remove year if present at start
        if (titleParts[0] && /^\d{4}$/.test(titleParts[0])) {
          titleParts.shift();
        }
        if (titleParts.length >= 2) {
          make = titleParts[0];
          model = titleParts.slice(1).join(' ');
          debug.push(`Make/Model from title: ${make} ${model}`);
        }
      }

      // Extract primary image
      let imageUrl = null;
      const heroImg = document.querySelector('[data-testid="hero-image"] img') ||
                      document.querySelector('[data-testid="gallery-image-0"] img') ||
                      document.querySelector('img[alt*="main"]') ||
                      document.querySelector('picture img') ||
                      document.querySelector('[class*="gallery"] img:first-of-type');

      if (heroImg) {
        imageUrl = heroImg.src || heroImg.getAttribute('data-src');
        debug.push(`Image found: ${imageUrl?.substring(0, 50)}...`);
      } else {
        debug.push('No image found');
      }

      // Extract mileage - try multiple methods
      let mileage = null;
      let mileageRaw = null;

      // Method 1: Find by "Mileage" label in previous sibling
      let mileageElement = null;
      const allParagraphs = Array.from(document.querySelectorAll('p'));
      const mileageLabel = allParagraphs.find(p => p.textContent.trim() === 'Mileage');

      if (mileageLabel) {
        // Look for next sibling or parent's next sibling
        let nextElement = mileageLabel.nextElementSibling;
        if (!nextElement && mileageLabel.parentElement) {
          nextElement = mileageLabel.parentElement.nextElementSibling;
        }
        if (!nextElement) {
          // Try finding the next p tag in the same container
          const container = mileageLabel.parentElement;
          if (container) {
            const containerPs = Array.from(container.querySelectorAll('p'));
            const labelIndex = containerPs.indexOf(mileageLabel);
            if (labelIndex >= 0 && labelIndex < containerPs.length - 1) {
              nextElement = containerPs[labelIndex + 1];
            }
          }
        }
        if (nextElement) {
          mileageElement = nextElement;
          debug.push(`Mileage found via "Mileage" label`);
        }
      }

      // Method 2: Fallback to data-testid or class
      if (!mileageElement) {
        mileageElement = document.querySelector('[data-testid="mileage"]') ||
                        document.querySelector('[class*="mileage"]') ||
                        Array.from(document.querySelectorAll('li, span, div, button')).find(el => {
                          const text = el.textContent.toLowerCase();
                          return /\d+[,\s]?\d*\s*miles/.test(text) && text.length < 50;
                        });
        if (mileageElement) {
          debug.push(`Mileage found via fallback selector`);
        }
      }

      if (mileageElement) {
        mileageRaw = mileageElement.textContent.trim();
        // Extract just the number from text like "40,085 miles" or "40200 miles"
        const mileageMatch = mileageRaw.match(/(\d{1,3})[,\s]?(\d{3})/);
        if (mileageMatch) {
          mileage = parseInt(mileageMatch[1] + mileageMatch[2], 10);
          debug.push(`Mileage parsed: "${mileageRaw}" → ${mileage}`);
        } else {
          const singleMatch = mileageRaw.match(/(\d+)/);
          if (singleMatch) {
            mileage = parseInt(singleMatch[1], 10);
            debug.push(`Mileage (single number): "${mileageRaw}" → ${mileage}`);
          }
        }
      } else {
        debug.push('Mileage: No element found');
      }

      // Extract year - try multiple methods
      let year = null;

      // Method 1: Find by "Registration" label which contains year
      const registrationLabel = allParagraphs.find(p => p.textContent.trim() === 'Registration');

      if (registrationLabel) {
        // Look for next sibling or parent's next sibling
        let nextElement = registrationLabel.nextElementSibling;
        if (!nextElement && registrationLabel.parentElement) {
          nextElement = registrationLabel.parentElement.nextElementSibling;
        }
        if (!nextElement) {
          // Try finding the next p tag in the same container
          const container = registrationLabel.parentElement;
          if (container) {
            const containerPs = Array.from(container.querySelectorAll('p'));
            const labelIndex = containerPs.indexOf(registrationLabel);
            if (labelIndex >= 0 && labelIndex < containerPs.length - 1) {
              nextElement = containerPs[labelIndex + 1];
            }
          }
        }
        if (nextElement) {
          const regText = nextElement.textContent.trim();
          // Extract year from text like "2021 (71 reg)"
          const yearMatch = regText.match(/\b(19|20)\d{2}\b/);
          if (yearMatch) {
            year = parseInt(yearMatch[0], 10);
            debug.push(`Year found via "Registration" label: "${regText}" → ${year}`);
          }
        }
      }

      // Method 2: Fallback to data-testid or class
      if (!year) {
        const yearElement = document.querySelector('[data-testid="year"]') ||
                           document.querySelector('[class*="year"]') ||
                           Array.from(document.querySelectorAll('li, span')).find(el => {
                             const text = el.textContent;
                             return /^(19|20)\d{2}$/.test(text.trim()) && text.length < 10;
                           });

        if (yearElement) {
          const yearText = yearElement.textContent.trim();
          const yearMatch = yearText.match(/\b(19|20)\d{2}\b/);
          if (yearMatch) {
            year = parseInt(yearMatch[0], 10);
            debug.push(`Year element found via fallback: "${yearText}" → ${year}`);
          }
        } else {
          debug.push('Year: No element found');
        }
      }

      // Extract registration
      const regElement = document.querySelector('[data-testid="registration"]');
      const registration = regElement ? regElement.textContent.trim() : null;
      debug.push(`Registration: ${registration}`);

      return {
        price,
        title,
        make,
        model,
        imageUrl,
        mileage,  // Now an integer
        mileageRaw,  // Original text for display
        year,  // Now an integer
        registration,
        url: window.location.href,
        debug: debug.join('\n')
      };
    }
  });

  return results[0].result;
}

// Display the car details
async function displayCarDetails() {
  const contentDiv = document.getElementById('content');

  try {
    const carDetails = await extractCarDetails();

    if (carDetails.error) {
      contentDiv.innerHTML = `<div class="error">${carDetails.error}</div>`;
      return;
    }

    if (!carDetails.price) {
      contentDiv.innerHTML = `<div class="error">Could not find price on this page. Make sure you're on a car listing page.</div>`;
      return;
    }

    // Get the stored car ID from chrome.storage
    let carId = null;
    try {
      const storage = await chrome.storage.local.get(['lastCarId']);
      carId = storage.lastCarId;
    } catch (error) {
      console.log('Could not retrieve car ID from storage:', error);
    }

    // Build the link URL - either to specific car or to homepage
    const linkUrl = carId
      ? `https://www.carpricetracker.com/car/${carId}/`
      : 'https://www.carpricetracker.com/';
    const linkText = carId
      ? '🚗 View This Car →'
      : '🚗 View Price Tracker →';

    contentDiv.innerHTML = `
      <div class="car-info">
        ${carDetails.imageUrl ? `<img src="${carDetails.imageUrl}" style="width: 100%; border-radius: 4px; margin-bottom: 10px;" alt="Car image">` : ''}
        ${carDetails.title ? `<div class="car-title">${carDetails.title}</div>` : ''}
        <div class="price">${carDetails.price}</div>
        ${carDetails.make ? `<div class="detail">🏭 Make: ${carDetails.make}</div>` : ''}
        ${carDetails.model ? `<div class="detail">🚗 Model: ${carDetails.model}</div>` : ''}
        ${carDetails.year ? `<div class="detail">📅 Year: ${carDetails.year}</div>` : ''}
        ${carDetails.mileage ? `<div class="detail">📏 Mileage: ${carDetails.mileage.toLocaleString()} miles</div>` : ''}
        ${carDetails.registration ? `<div class="detail">🔢 Reg: ${carDetails.registration}</div>` : ''}
      </div>
      <a href="${linkUrl}" target="_blank" class="site-link">${linkText}</a>
      <details style="margin-top: 10px;">
        <summary style="cursor: pointer; color: #7f8c8d; font-size: 12px;">Debug Log</summary>
        <pre style="font-size: 10px; background: #f8f9fa; padding: 8px; border-radius: 4px; max-height: 200px; overflow-y: auto; margin-top: 5px;">${carDetails.debug || 'No debug info'}</pre>
      </details>
    `;

  } catch (error) {
    contentDiv.innerHTML = `<div class="error">Error: ${error.message}</div>`;
  }
}

// Run when popup opens
displayCarDetails();
